<?php

function handleMemberCommands($chat_id, $user_id, $text, $msg_id, $reply_to = null) {
    $text = trim($text);
    $role = getUserRole($chat_id, $user_id);

    // استخراج target_id و reason
    $target_id = null;
    $reason = null;

    if ($reply_to && isset($reply_to['from']['id']) && is_numeric($reply_to['from']['id'])) {
        $target_id = (int)$reply_to['from']['id'];
        // دلیل = متن اصلی دستور
        $reason = $text;
    } else {
        // تحلیل متن برای یوزرنیم یا آیدی عددی
        if (preg_match('/^.*?(@[a-zA-Z0-9_]+)/', $text, $m)) {
            $username = substr($m[1], 1);
            $target_id = getUserIdByUsername($chat_id, $username);
            $reason = preg_replace('/^.*?(@[a-zA-Z0-9_]+)\s*/', '', $text);
        } elseif (preg_match('/^.*?(\d{7,15})/', $text, $m)) {
            $target_id = (int)$m[1];
            $reason = preg_replace('/^.*?(\d{7,15})\s*/', '', $text);
        }
    }

    // === دستورات عمومی (بدون target) ===
    if (in_array($text, ['ایدی', 'اطلاعات', 'info'])) {
        if ($role === 'member') return false;
        $tid = $target_id ?? $user_id;
        if (!$tid || !is_numeric($tid)) return false;
        showUserInfo($chat_id, $tid, $msg_id, $msg_id);
        return true;
    }

    if (in_array($text, ['لقب من', 'لقبم'])) {
        if ($role === 'member') return false;
        $nickname = getUserNickname($chat_id, $user_id);
        if ($nickname) {
            sendMessageWithAutoDelete($chat_id, "📖 لقب شما: <b>$nickname</b>", $msg_id);
        } else {
            sendMessageWithAutoDelete($chat_id, "❌ شما لقبی ندارید.", $msg_id);
        }
        return true;
    }

    if (in_array($text, ['تگ', 'تگ اعضا', 'منشن'])) {
        if (!canManageLocks($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی تگ کردن اعضا را ندارید.", $msg_id);
            return true;
        }
        showTagPanel($chat_id, $user_id, $msg_id);
        return true;
    }

    // === بررسی دستوراتی که به target نیاز دارند ===
    $admin_commands = ['بن', '/ban', 'حذف بن', '/unban', 'سکوت', 'mute', 'باز کردن آن', 'unmute', 'اخطار', 'warn', 'حذف اخطار', 'remove warn'];
    if ($target_id === null && in_array($text, $admin_commands)) {
        sendMessageWithAutoDelete($chat_id, "❌ لطفاً روی پیام کاربر ریپلای کنید یا یوزرنیم/آیدی را وارد کنید.", $msg_id);
        return true;
    }

    // === بن ===
    if (in_array($text, ['/ban', 'بن'])) {
        if (!canBanUsers($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی بن کردن را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        if (!isUserBannable($chat_id, $target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ این کاربر قابل بن کردن نیست.", $msg_id);
            return true;
        }
        $stmt = db()->prepare("SELECT 1 FROM banned_users WHERE chat_id = ? AND user_id = ?");
        $stmt->execute([$chat_id, $target_id]);
        if ($stmt->fetch()) {
            sendMessageWithAutoDelete($chat_id, "✅ این کاربر در لیست بن وجود دارد.", $msg_id);
            return true;
        }
        $res = sendTelegram('banChatMember', ['chat_id' => $chat_id, 'user_id' => $target_id]);
        if (!($res['ok'] ?? false)) {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در بن کردن. ربات ممکن است ادمین نباشد.", $msg_id);
            return true;
        }
        logBannedUser($chat_id, $target_id, $user_id);
        $safe_mention = formatUserMention($target_id);
        sendMessageWithAutoDelete($chat_id, "🚫 کاربر $safe_mention از گروه بن شد.", $msg_id);
        return true;
    }

    // === آنبن ===
    if (in_array($text, ['/unban', 'حذف بن', 'آنبن'])) {
        if (!canBanUsers($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی آنبن کردن را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        $stmt = db()->prepare("SELECT 1 FROM banned_users WHERE chat_id = ? AND user_id = ?");
        $stmt->execute([$chat_id, $target_id]);
        if (!$stmt->fetch()) {
            sendMessageWithAutoDelete($chat_id, "✅ این کاربر در لیست بن وجود ندارد.", $msg_id);
            return true;
        }
        $res = sendTelegram('unbanChatMember', ['chat_id' => $chat_id, 'user_id' => $target_id]);
        if (!($res['ok'] ?? false)) {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در آنبن کردن.", $msg_id);
            return true;
        }
        unlogBannedUser($chat_id, $target_id);
        $safe_mention = formatUserMention($target_id);
        sendMessageWithAutoDelete($chat_id, "✅ کاربر $safe_mention از گروه آزاد شد.", $msg_id);
        return true;
    }

    // === سکوت ===
    if (in_array($text, ['سکوت', 'mute'])) {
        if (!canBanUsers($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی سکوت کردن را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        if (!isUserBannable($chat_id, $target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ این کاربر قابل سکوت کردن نیست.", $msg_id);
            return true;
        }
        $res = sendTelegram('restrictChatMember', [
            'chat_id' => $chat_id,
            'user_id' => $target_id,
            'permissions' => [
                'can_send_messages' => false,
                'can_send_media_messages' => false,
                'can_send_polls' => false,
                'can_send_other_messages' => false,
                'can_add_web_page_previews' => false,
                'can_change_info' => false,
                'can_invite_users' => true,
                'can_pin_messages' => false
            ]
        ]);
        if (!($res['ok'] ?? false)) {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در اعمال سکوت.", $msg_id);
            return true;
        }
        sendMessageWithAutoDelete($chat_id, "🔇 کاربر " . formatUserMention($target_id) . " در گروه سکوت شد.", $msg_id);
        return true;
    }

    // === باز کردن سکوت ===
    if (in_array($text, ['باز کردن آن', 'آزاد', 'unmute', 'باز کردن سکوت'])) {
        if (!canBanUsers($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی باز کردن سکوت را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        $res = sendTelegram('restrictChatMember', [
            'chat_id' => $chat_id,
            'user_id' => $target_id,
            'permissions' => [
                'can_send_messages' => true,
                'can_send_media_messages' => true,
                'can_send_polls' => true,
                'can_send_other_messages' => true,
                'can_add_web_page_previews' => true,
                'can_change_info' => true,
                'can_invite_users' => true,
                'can_pin_messages' => true
            ]
        ]);
        if (!($res['ok'] ?? false)) {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در باز کردن سکوت.", $msg_id);
            return true;
        }
        sendMessageWithAutoDelete($chat_id, "🔊 کاربر " . formatUserMention($target_id) . " از سکوت خارج شد.", $msg_id);
        return true;
    }

    // === اخطار ===
    if (in_array($text, ['اخطار', 'warn'])) {
        if (!canBanUsers($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی اخطار دادن را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        if (!isUserBannable($chat_id, $target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ به این کاربر نمی‌توان اخطار داد.", $msg_id);
            return true;
        }

        $reason_clean = null;
        if ($reason) {
            $cmd_removed = preg_replace('/^(اخطار|warn)\s*/i', '', $reason);
            $reason_clean = substr(trim($cmd_removed), 0, 200);
        }

        db()->prepare("INSERT INTO warnings (chat_id, user_id, warned_by, reason) VALUES (?, ?, ?, ?)")->execute([
            $chat_id, $target_id, $user_id, $reason_clean
        ]);

        $stmt = db()->prepare("SELECT COUNT(*) FROM warnings WHERE chat_id = ? AND user_id = ?");
        $stmt->execute([$chat_id, $target_id]);
        $warn_count = (int)$stmt->fetchColumn();

        $safe_mention = formatUserMention($target_id);
        if ($warn_count >= 5) {
            sendTelegram('restrictChatMember', [
                'chat_id' => $chat_id,
                'user_id' => $target_id,
                'permissions' => [
                    'can_send_messages' => false,
                    'can_send_media_messages' => false,
                    'can_send_polls' => false,
                    'can_send_other_messages' => false,
                    'can_add_web_page_previews' => false,
                    'can_change_info' => false,
                    'can_invite_users' => true,
                    'can_pin_messages' => false
                ]
            ]);
            sendMessageWithAutoDelete($chat_id, "🔇 کاربر $safe_mention به دلیل دریافت ۵ اخطار، در گروه سکوت شد.", $msg_id);
        } else {
            $msg = "⚠️ به کاربر $safe_mention اخطار داده شد.\n📊 تعداد اخطارها: $warn_count/5";
            if ($reason_clean) $msg .= "\n📝 دلیل: $reason_clean";
            sendMessageWithAutoDelete($chat_id, $msg, $msg_id);
        }
        return true;
    }

    // === حذف اخطار ===
    if (in_array($text, ['حذف اخطار', 'remove warn', 'clear warn'])) {
        if (!canBanUsers($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی حذف اخطار را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }

        db()->prepare("DELETE FROM warnings WHERE chat_id = ? AND user_id = ?")->execute([$chat_id, $target_id]);
        sendMessageWithAutoDelete($chat_id, "✅ تمام اخطارهای کاربر " . formatUserMention($target_id) . " پاک شد.", $msg_id);
        return true;
    }

    // === قفل‌ها ===
    if (preg_match('/^قفل\s+(.+)$/iu', $text, $m)) {
        if (!canManageLocks($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی تنظیم قفل‌ها را ندارید.", $msg_id);
            return true;
        }
        $type_fa = trim($m[1]);
        $lock_types = [
            'لینک' => 'link', 'استیکر' => 'sticker', 'عکس' => 'photo',
            'ویدیو' => 'video', 'فایل' => 'document', 'صدا' => 'voice',
            'گیف' => 'animation', 'مخاطب' => 'contact', 'لوکیشن' => 'location',
            'فوروارد' => 'forward'
        ];
        $type_en = $lock_types[$type_fa] ?? null;
        if (!$type_en) {
            $available = implode('، ', array_keys($lock_types));
            sendMessageWithAutoDelete($chat_id, "❌ نوع قفل نامعتبر است.\n\nانواع موجود: $available", $msg_id);
            return true;
        }
        if (isLocked($chat_id, $type_en)) {
            sendMessageWithAutoDelete($chat_id, "🔒 قفل $type_fa در حال حاضر فعال است!", $msg_id);
        } else {
            setLock($chat_id, $type_en, true);
            sendMessageWithAutoDelete($chat_id, "✅ قفل $type_fa با موفقیت فعال شد.", $msg_id);
        }
        return true;
    }

    if (preg_match('/^بازکردن\s+(.+)$/iu', $text, $m)) {
        if (!canManageLocks($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی تنظیم قفل‌ها را ندارید.", $msg_id);
            return true;
        }
        $type_fa = trim($m[1]);
        $lock_types = [
            'لینک' => 'link', 'استیکر' => 'sticker', 'عکس' => 'photo',
            'ویدیو' => 'video', 'فایل' => 'document', 'صدا' => 'voice',
            'گیف' => 'animation', 'مخاطب' => 'contact', 'لوکیشن' => 'location',
            'فوروارد' => 'forward'
        ];
        $type_en = $lock_types[$type_fa] ?? null;
        if (!$type_en) {
            $available = implode('، ', array_keys($lock_types));
            sendMessageWithAutoDelete($chat_id, "❌ نوع قفل نامعتبر است.\n\nانواع موجود: $available", $msg_id);
            return true;
        }
        if (!isLocked($chat_id, $type_en)) {
            sendMessageWithAutoDelete($chat_id, "✅ قفل $type_fa با موفقیت غیرفعال شد.", $msg_id);
        } else {
            setLock($chat_id, $type_en, false);
            sendMessageWithAutoDelete($chat_id, "✅ قفل $type_fa با موفقیت غیرفعال شد.", $msg_id);
        }
        return true;
    }

    // === دستورات لقب و سنجاق ===
    if (preg_match('/^تنظیم لقب\s+(.+)$/u', $text, $m)) {
        if (!canManageLocks($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی تنظیم لقب را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        $nickname = trim($m[1]);
        if (mb_strlen($nickname) > 50) {
            sendMessageWithAutoDelete($chat_id, "❌ لقب نباید بیشتر از 50 کاراکتر باشد.", $msg_id);
            return true;
        }
        setUserNickname($chat_id, $target_id, $nickname);
        $safe_mention = formatUserMention($target_id);
        sendMessageWithAutoDelete($chat_id, "✅ لقب کاربر $safe_mention تنظیم شد:\n\n📖 <b>$nickname</b>", $msg_id);
        return true;
    }

    if (in_array($text, ['حذف لقب', 'پاک کردن لقب'])) {
        if (!canManageLocks($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی حذف لقب را ندارید.", $msg_id);
            return true;
        }
        if (!$target_id || !is_numeric($target_id)) {
            sendMessageWithAutoDelete($chat_id, "❌ کاربر مورد نظر یافت نشد.", $msg_id);
            return true;
        }
        removeUserNickname($chat_id, $target_id);
        $safe_mention = formatUserMention($target_id);
        sendMessageWithAutoDelete($chat_id, "✅ لقب کاربر $safe_mention حذف شد.", $msg_id);
        return true;
    }

    if (in_array($text, ['سنجاق', 'پین', 'pin'])) {
        if (!$reply_to || !isset($reply_to['message_id'])) {
            sendMessageWithAutoDelete($chat_id, "❌ روی پیام مورد نظر ریپلای کنید.", $msg_id);
            return true;
        }
        if (!canPinMessages($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "⛔️ شما دسترسی سنجاق ندارید.", $msg_id);
            return true;
        }
        $res = sendTelegram('pinChatMessage', ['chat_id' => $chat_id, 'message_id' => $reply_to['message_id'], 'disable_notification' => false]);
        if ($res['ok'] ?? false) {
            sendMessageWithAutoDelete($chat_id, "📌 پیام با موفقیت سنجاق شد.", $msg_id);
        } else {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در سنجاق کردن پیام.", $msg_id);
        }
        return true;
    }

    if (in_array($text, ['حذف سنجاق', 'آنپین', 'unpin'])) {
        if (!canPinMessages($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "⛔️ شما دسترسی مدیریت سنجاق ندارید.", $msg_id);
            return true;
        }
        $params = ['chat_id' => $chat_id];
        if ($reply_to) $params['message_id'] = $reply_to['message_id'];
        $res = sendTelegram('unpinChatMessage', $params);
        if ($res['ok'] ?? false) {
            $txt = $reply_to ? "✅ پیام از سنجاق برداشته شد." : "✅ آخرین پیام سنجاق برداشته شد.";
            sendMessageWithAutoDelete($chat_id, $txt, $msg_id);
        } else {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در برداشتن سنجاق.", $msg_id);
        }
        return true;
    }

    if (in_array($text, ['حذف همه سنجاق', 'پاک کردن پین‌ها'])) {
        if (!canPinMessages($chat_id, $user_id)) {
            sendMessageWithAutoDelete($chat_id, "⛔️ شما دسترسی مدیریت سنجاق ندارید.", $msg_id);
            return true;
        }
        $res = sendTelegram('unpinAllChatMessages', ['chat_id' => $chat_id]);
        if ($res['ok'] ?? false) {
            sendMessageWithAutoDelete($chat_id, "✅ تمام سنجاق‌ها برداشته شدند.", $msg_id);
        } else {
            sendMessageWithAutoDelete($chat_id, "❌ خطا در برداشتن سنجاق‌ها.", $msg_id);
        }
        return true;
    }

    return false;
}

// === توابع کمکی ===

function getUserIdByUsername($chat_id, $username) {
    $stmt = db()->prepare("SELECT user_id FROM user_cache WHERE username = ?");
    $stmt->execute([$username]);
    $row = $stmt->fetch();
    if ($row) return (int)$row['user_id'];
    return null;
}

function logBannedUser($chat_id, $user_id, $banned_by) {
    try {
        db()->prepare("INSERT INTO banned_users (chat_id, user_id, banned_by, banned_at) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE banned_by = ?, banned_at = NOW()")
            ->execute([$chat_id, $user_id, $banned_by, $banned_by]);
    } catch (Exception $e) {
        log_debug("خطا در ثبت کاربر بن شده: " . $e->getMessage());
    }
}

function unlogBannedUser($chat_id, $user_id) {
    try {
        db()->prepare("DELETE FROM banned_users WHERE chat_id = ? AND user_id = ?")->execute([$chat_id, $user_id]);
    } catch (Exception $e) {
        log_debug("خطا در حذف ثبت کاربر بن شده: " . $e->getMessage());
    }
}

function safeUserMention($user_id) {
    if (!is_numeric($user_id)) return "کاربر نامشخص";
    $stmt = db()->prepare("SELECT username, first_name FROM user_cache WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    if ($user && $user['username']) {
        return "@{$user['username']}";
    }
    return "<a href='tg://user?id=$user_id'>" . htmlspecialchars($user['first_name'] ?? "کاربر") . "</a>";
}

function showTagPanel($chat_id, $user_id, $msg_id) {
    $hash = bin2hex(random_bytes(8));
    $panel_data = [
        'chat_id' => $chat_id,
        'issuer_id' => $user_id,
        'reply_to_msg_id' => $msg_id,
        'type' => 'tag',
        'created_at' => time()
    ];
    setCache("panel_$hash", json_encode($panel_data), 300);

    $keyboard = [
        'inline_keyboard' => [
            [['text' => '👥 تگ همه اعضا', 'callback_data' => "tag_members_$hash"]],
            [['text' => '👨‍💼 تگ مدیران', 'callback_data' => "tag_admins_$hash"]],
            [['text' => '❌ بستن', 'callback_data' => "tag_close_$hash"]]
        ]
    ];
    
    $result = sendTelegram('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "🏷️ <b>پنل تگ اعضا</b>\n\nیک گزینه را انتخاب کنید:",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode($keyboard),
        'reply_to_message_id' => $msg_id
    ]);
    
    if ($result && isset($result['result']['message_id'])) {
        scheduleMessageDeletion($chat_id, $result['result']['message_id'], 60);
    }
}

function tagAllMembers($chat_id, $reply_to_msg_id) {
    try {
        $stmt = db()->prepare("
            SELECT DISTINCT user_id 
            FROM messages 
            WHERE chat_id = ? 
              AND user_id > 1000000000
            ORDER BY sent_at DESC 
            LIMIT 200
        ");
        $stmt->execute([$chat_id]);
        $member_ids = $stmt->fetchAll(PDO::FETCH_COLUMN);
        if (empty($member_ids)) {
            sendMessageWithAutoDelete($chat_id, "❌ هیچ عضوی برای تگ وجود ندارد.", $reply_to_msg_id);
            return;
        }
        $mentions = "👥 <b>تگ اعضا:</b>\n\n";
        $max_length = 4000;
        foreach ($member_ids as $uid) {
            if (!is_numeric($uid) || $uid <= 1000000000) continue;
            $safe_mention = formatUserMention($uid);
            $new_line = "• $safe_mention\n";
            if (strlen($mentions . $new_line) > $max_length) {
                $mentions .= "\n... و " . (count($member_ids) - substr_count($mentions, "\n") + 2) . " عضو دیگر.";
                break;
            }
            $mentions .= $new_line;
        }
        $mentions .= "\n📊 تعداد: " . count($member_ids) . " نفر";
        sendTelegram('sendMessage', [
            'chat_id' => $chat_id,
            'text' => $mentions,
            'parse_mode' => 'HTML',
            'reply_to_message_id' => $reply_to_msg_id
        ]);
    } catch (Exception $e) {
        log_debug("خطا در تگ اعضا: " . $e->getMessage());
        sendMessageWithAutoDelete($chat_id, "❌ خطا در تگ اعضا.", $reply_to_msg_id);
    }
}

// === توابع showUserInfo, setUserNickname, etc. (بدون تغییر) ===
function showUserInfo($chat_id, $target_id, $reply_to_msg_id, $target_msg_id = null) {
    try {
        $stmt = db()->prepare("SELECT first_name, last_name, username FROM user_cache WHERE user_id = ?");
        $stmt->execute([$target_id]);
        $user = $stmt->fetch();
        if (!$user) {
            $user_info = sendTelegram('getChatMember', ['chat_id' => $chat_id, 'user_id' => $target_id]);
            if ($user_info['ok'] ?? false) {
                $member = $user_info['result']['user'] ?? [];
                $first_name = $member['first_name'] ?? 'کاربر';
                $last_name = $member['last_name'] ?? '';
                $username = $member['username'] ?? '';
                cacheUser($target_id, $first_name, $last_name, $username);
                $full_name = $first_name . ($last_name ? ' ' . $last_name : '');
            } else {
                sendMessageWithAutoDelete($chat_id, "❌ اطلاعات کاربر یافت نشد.", $reply_to_msg_id);
                return;
            }
        } else {
            $full_name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
            $username = $user['username'] ?? '';
        }
        $username_text = $username ? "@$username" : "ندارد";
        $nickname = getUserNickname($chat_id, $target_id) ?: "ندارد";

        $role = getUserRole($chat_id, $target_id);
        $real_owner_id = getGroupOwnerFromChat($chat_id);
        
        if ((int)$target_id === OWNER_ID) {
            $role_text = "👑 مالک ربات";
        } elseif ($real_owner_id && (int)$target_id === (int)$real_owner_id) {
            $role_text = "👑 سازنده گروه";
        } elseif ($role === 'owner_assigned') {
            $role_text = "👑 مالک اختصاصی";
        } elseif ($role === 'vip') {
            $role_text = "🌟 ویژه";
        } elseif (is_array($role) && $role['role'] === 'admin') {
            $role_text = "👨‍💼 مدیر";
        } else {
            $role_text = "👤 عضو";
        }

        $stmt = db()->prepare("SELECT COUNT(*) FROM messages WHERE chat_id = ? AND user_id = ? AND DATE(sent_at) = CURDATE()");
        $stmt->execute([$chat_id, $target_id]);
        $today_messages = (int)$stmt->fetchColumn();
        
        $stmt = db()->prepare("SELECT COUNT(*) FROM messages WHERE chat_id = ? AND user_id = ?");
        $stmt->execute([$chat_id, $target_id]);
        $user_total = (int)$stmt->fetchColumn();
        
        $rank = "—";
        if ($user_total > 0) {
            $stmt = db()->prepare("SELECT COUNT(*) FROM (SELECT user_id FROM messages WHERE chat_id = ? GROUP BY user_id HAVING COUNT(*) > ?) t");
            $stmt->execute([$chat_id, $user_total]);
            $rank = "#" . ((int)$stmt->fetchColumn() + 1);
        }

        $info_text = "📷 <b>اطلاعات کاربر</b>\n";
        $info_text .= "├─ نام: " . formatUserMention($target_id) . "\n";
        $info_text .= "├─ آیدی: <code>$target_id</code>\n";
        $info_text .= "├─ یوزرنیم: $username_text\n";
        $info_text .= "├─ لقب: $nickname\n";
        $info_text .= "├─ نقش: $role_text\n";
        $info_text .= "\n───┤ آمار ├───\n";
        $info_text .= "├─ امروز: $today_messages\n";
        $info_text .= "├─ کل: $user_total\n";
        $info_text .= "├─ رتبه: $rank";
        
        sendTelegram('sendMessage', [
            'chat_id' => $chat_id,
            'text' => $info_text,
            'parse_mode' => 'HTML',
            'reply_to_message_id' => $target_msg_id ?: $reply_to_msg_id
        ]);
    } catch (Exception $e) {
        log_debug("خطا در اطلاعات کاربر: " . $e->getMessage());
        sendMessageWithAutoDelete($chat_id, "❌ خطا در دریافت اطلاعات.", $reply_to_msg_id);
    }
}

function setUserNickname($chat_id, $user_id, $nickname) {
    db()->prepare("INSERT INTO user_nicknames (chat_id, user_id, nickname, set_at) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE nickname = ?, set_at = NOW()")->execute([$chat_id, $user_id, $nickname, $nickname]);
}

function removeUserNickname($chat_id, $user_id) {
    db()->prepare("DELETE FROM user_nicknames WHERE chat_id = ? AND user_id = ?")->execute([$chat_id, $user_id]);
}

function getUserNickname($chat_id, $user_id) {
    $stmt = db()->prepare("SELECT nickname FROM user_nicknames WHERE chat_id = ? AND user_id = ?");
    $stmt->execute([$chat_id, $user_id]);
    return $stmt->fetchColumn();
}