<?php
/**
 * ═══════════════════════════════════════════════════════
 * 👤 PV - سیستم مدیریت پنل کاربری و آنتی اسپم
 * ═══════════════════════════════════════════════════════
 */

require_once 'config.php';
require_once 'functions.php';

// تنظیمات ظاهری
define('PV_BANNER', 'https://khan9.ir/Ebi1/a.jpg');

/**
 * 🛡️ بررسی آنتی اسپم برای PV
 * اگر کاربر اسپم کند true برمی‌گرداند
 */
function isPvSpam(int $userId): bool {
    // 1. اگر کاربر در حال نوشتن نجوا است، اسپم را چک نکن (اجازه بده بنویسه)
    // بررسی جدول user_states که در فایل n.php مدیریت می‌شود
    $db = Database::instance()->pdo();
    $stmt = $db->prepare("SELECT step FROM user_states WHERE user_id = ?");
    $stmt->execute([$userId]);
    $state = $stmt->fetchColumn();
    
    if ($state === 'awaiting_whisper') {
        return false; // کاربر در حال نوشتن نجواست، اسپم نیست
    }

    // 2. بررسی زمان آخرین درخواست (استفاده از جدول تمپ برای سرعت)
    // محدودیت: 1 درخواست در هر 1.5 ثانیه
    $stmt = $db->prepare("SELECT data_value FROM mir_temp_data WHERE user_id = ? AND data_key = 'last_pv_action'");
    $stmt->execute([$userId]);
    $lastTime = $stmt->fetchColumn();

    $currentTime = microtime(true);

    if ($lastTime && ($currentTime - (float)$lastTime) < 1.5) {
        return true; // اسپم تشخیص داده شد
    }

    // بروزرسانی زمان آخرین فعالیت
    $stmt = $db->prepare("
        INSERT INTO mir_temp_data (group_id, user_id, data_key, data_value) 
        VALUES (0, ?, 'last_pv_action', ?) 
        ON DUPLICATE KEY UPDATE data_value = ?
    ");
    $stmt->execute([$userId, $currentTime, $currentTime]);

    return false;
}

/**
 * ⌨️ کیبورد اصلی پنل
 */
function getPvKeyboard() {
    $botUsername = str_replace('@', '', BOT_USERNAME);
    
    return [
        'inline_keyboard' => [
            [['text' => "➕ افزودن من به گروه", 'url' => "https://t.me/$botUsername?startgroup=new"]],
            [
                ['text' => "📚 راهنما", 'callback_data' => "pv_help"],
                ['text' => "⚙️ نصب", 'callback_data' => "pv_install"]
            ],
            [
                ['text' => "👤 حساب کاربری", 'callback_data' => "pv_account"],
                ['text' => "👨‍💻 سازنده", 'callback_data' => "pv_creator"]
            ]
        ]
    ];
}

/**
 * 📝 متن اصلی خوش‌آمدگویی
 */
function getPvMainText($name) {
    $txt = "╔═══════════════════╗\n";
    $txt .= "║  🤖 <b>خوش آمدید</b>  ║\n";
    $txt .= "╚═══════════════════╝\n\n";
    $txt .= "👋 سلام <b>{$name}</b> عزیز!\n\n";
    $txt .= "من یک ربات پیشرفته مدیریت گروه هستم.\n\n";
    $txt .= "✨ <b>امکانات من:</b>\n";
    $txt .= "🛡️ قفل‌های پیشرفته و آنتی لینک\n";
    $txt .= "🔞 آنتی پورن هوشمند (Google AI)\n";
    $txt .= "🤫 سیستم نجوا و پیام محرمانه\n";
    $txt .= "👮 مدیریت مقامات و سطح دسترسی\n\n";
    $txt .= "👇 برای شروع یکی از گزینه‌ها را انتخاب کنید:";
    return $txt;
}

/**
 * 🚀 هندلر استارت (PV)
 */
function handlePvStart($update) {
    $message = $update['message'];
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    $name = htmlspecialchars($message['from']['first_name']);

    // 🚫 بررسی آنتی اسپم
    // نکته: اگر کاربر لینک نجوا زده باشد، Ebi.php قبل از رسیدن به اینجا آن را پردازش کرده
    // پس اینجا فقط استارت‌های خالی می‌آیند
    if (isPvSpam($userId)) {
        // اختیاری: می‌توانید پیام "لطفا صبر کنید" بفرستید یا کلا نادیده بگیرید
        return; 
    }

    // ارسال عکس با کپشن (متد sendPhoto در functions.php وجود نداشت، اینجا دستی هندل می‌کنیم)
    $params = [
        'chat_id' => $chatId,
        'photo' => PV_BANNER,
        'caption' => getPvMainText($name),
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(getPvKeyboard())
    ];
    
    sendRequest('sendPhoto', $params);
}

/**
 * 🔄 هندلر دکمه‌های شیشه‌ای (Callback)
 */
function handlePvCallback($update) {
    $cb = $update['callback_query'];
    $data = $cb['data'];
    $chatId = $cb['message']['chat']['id'];
    $msgId = $cb['message']['message_id'];
    $userId = $cb['from']['id'];
    $name = htmlspecialchars($cb['from']['first_name']);

    // دکمه بازگشت مشترک
    $backBtn = [['text' => "🔙 بازگشت به منو", 'callback_data' => "pv_home"]];

    // 1. بازگشت به خانه
    if ($data == 'pv_home') {
        editMessageCaption(
            $chatId, 
            $msgId, 
            getPvMainText($name), 
            getPvKeyboard()
        );
        answerCallbackQuery($cb['id']);
    }

    // 2. راهنما
    elseif ($data == 'pv_help') {
        $txt = "╔═══════════════════╗\n";
        $txt .= "║  📚 <b>راهنمای ربات</b>  ║\n";
        $txt .= "╚═══════════════════╝\n\n";
        $txt .= "🔹 <b>شروع کار:</b>\n";
        $txt .= "ربات را به گروه ببرید و دستور <code>!Add</code> یا <code>نصب</code> را ارسال کنید.\n\n";
        $txt .= "🔹 <b>مدیریت قفل‌ها:</b>\n";
        $txt .= "دستور: <code>قفل‌ها</code> یا <code>پنل قفل</code>\n\n";
        $txt .= "🔹 <b>ارسال نجوا:</b>\n";
        $txt .= "روی پیام کاربر ریپلای کنید و بنویسید: <code>نجوا</code>\n\n";
        $txt .= "🔹 <b>مدیریت کاربر:</b>\n";
        $txt .= "روی کاربر ریپلای کنید و بنویسید: <code>پنل کاربر</code>\n\n";
        $txt .= "🔹 <b>ابزارهای ME:</b>\n";
        $txt .= "دستورات: <code>آیدی</code>، <code>تگ</code>، <code>ساعت</code>\n\n";
        $txt .= "⚡️ برای دیدن دستورات کامل، ربات را به گروه اضافه کنید.";
        
        editMessageCaption($chatId, $msgId, $txt, ['inline_keyboard' => [$backBtn]]);
        answerCallbackQuery($cb['id']);
    }

    // 3. راهنمای نصب
    elseif ($data == 'pv_install') {
        $txt = "╔═══════════════════╗\n";
        $txt .= "║  ⚙️ <b>راهنمای نصب</b>  ║\n";
        $txt .= "╚═══════════════════╝\n\n";
        $txt .= "<b>مراحل فعال‌سازی در گروه:</b>\n\n";
        $txt .= "1️⃣ ربات را به گروه خود اضافه کنید.\n";
        $txt .= "2️⃣ ربات را <b>ادمین (Admin)</b> کنید.\n";
        $txt .= "3️⃣ دستور <code>!Add</code> را در گروه ارسال کنید.\n\n";
        $txt .= "💡 <b>نکته مهم:</b>\n";
        $txt .= "برای نصب، باید مالک گروه یا سودوی ربات باشید.";
        
        editMessageCaption($chatId, $msgId, $txt, ['inline_keyboard' => [$backBtn]]);
        answerCallbackQuery($cb['id']);
    }

    // 4. اطلاعات سازنده
    elseif ($data == 'pv_creator') {
        $txt = "╔═══════════════════╗\n";
        $txt .= "║  👨‍💻 <b>درباره سازنده</b>  ║\n";
        $txt .= "╚═══════════════════╝\n\n";
        $txt .= "💎 <b>توسعه دهنده:</b> تیم خان بابا\n";
        $txt .= "📅 <b>نسخه ربات:</b> " . BOT_META['version'] . "\n";
        $txt .= "🛡 <b>وضعیت سیستم:</b> آنلاین ✅\n\n";
        $txt .= "✉️ برای پشتیبانی به گروه پشتیبانی مراجعه کنید.";
        
        editMessageCaption($chatId, $msgId, $txt, ['inline_keyboard' => [$backBtn]]);
        answerCallbackQuery($cb['id']);
    }

    // 5. حساب کاربری (اصلاح شده با جداول دیتابیس)
    elseif ($data == 'pv_account') {
        $db = Database::instance()->pdo();
        
        // شمارش گروه‌هایی که کاربر سازنده آنهاست (جدول group_admins)
        $stmt = $db->prepare("SELECT COUNT(*) FROM group_admins WHERE user_id = ? AND is_creator = 1");
        $stmt->execute([$userId]);
        $totalGroups = $stmt->fetchColumn();

        // شمارش گروه‌های فعال این کاربر (ترکیب با groups_data)
        $stmt = $db->prepare("
            SELECT COUNT(*) FROM group_admins ga
            JOIN groups_data gd ON ga.group_id = gd.group_id
            WHERE ga.user_id = ? AND ga.is_creator = 1 AND gd.is_active = 1
        ");
        $stmt->execute([$userId]);
        $activeGroups = $stmt->fetchColumn();

        $expiredGroups = $totalGroups - $activeGroups;

        $txt = "╔═══════════════════╗\n";
        $txt .= "║  👤 <b>حساب کاربری</b>  ║\n";
        $txt .= "╚═══════════════════╝\n\n";
        $txt .= "📛 <b>نام شما:</b> {$name}\n";
        $txt .= "🆔 <b>آیدی عددی:</b> <code>{$userId}</code>\n\n";
        $txt .= "📊 <b>آمار گروه‌های شما:</b>\n";
        $txt .= "📂 کل گروه‌های نصب شده: <b>{$totalGroups}</b>\n";
        $txt .= "✅ گروه‌های فعال: <b>{$activeGroups}</b>\n";
        $txt .= "⚠️ گروه‌های منقضی: <b>{$expiredGroups}</b>";

        editMessageCaption($chatId, $msgId, $txt, ['inline_keyboard' => [$backBtn]]);
        answerCallbackQuery($cb['id']);
    }
}

/**
 * 🛠️ تابع کمکی برای ویرایش کپشن عکس
 */
function editMessageCaption($chatId, $messageId, $caption, $keyboard = null) {
    $params = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'caption' => $caption,
        'parse_mode' => 'HTML'
    ];
    
    if ($keyboard) {
        $params['reply_markup'] = json_encode($keyboard);
    }
    
    return sendRequest('editMessageCaption', $params);
}
?>