<?php
/**
 * ═══════════════════════════════════════════════════════
 * 🛠️ توابع کمکی مشترک - نسخه کامل و اصلاح شده
 * ═══════════════════════════════════════════════════════
 */

// ─────────────────────────────────────────────────────────
// 📡 توابع ارتباط با تلگرام
// ─────────────────────────────────────────────────────────

/**
 * ارسال درخواست به API تلگرام
 */
function sendRequest(string $method, array $params = []): ?array
{
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/{$method}";
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $params,
        CURLOPT_TIMEOUT => 15,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response) {
        return json_decode($response, true);
    }
    
    return null;
}

/**
 * ارسال پیام متنی
 */
function sendMessage(
    int|string $chatId,
    string $text,
    ?array $keyboard = null,
    ?int $replyTo = null
): ?array {
    $params = [
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($keyboard !== null) {
        $params['reply_markup'] = json_encode($keyboard);
    }
    
    if ($replyTo !== null) {
        $params['reply_to_message_id'] = $replyTo;
    }
    
    return sendRequest('sendMessage', $params);
}

/**
 * ویرایش پیام
 */
function editMessage(
    int|string $chatId,
    int $messageId,
    string $text,
    ?array $keyboard = null
): ?array {
    $params = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($keyboard !== null) {
        $params['reply_markup'] = json_encode($keyboard);
    }
    
    return sendRequest('editMessageText', $params);
}

/**
 * حذف پیام
 */
function deleteMessage(int|string $chatId, int $messageId): bool
{
    $result = sendRequest('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
    
    return isset($result['ok']) && $result['ok'] === true;
}

/**
 * پاسخ به کالبک کوئری
 */
function answerCallbackQuery(
    string $callbackId,
    string $text = '',
    bool $showAlert = false
): void {
    sendRequest('answerCallbackQuery', [
        'callback_query_id' => $callbackId,
        'text' => $text,
        'show_alert' => $showAlert
    ]);
}

/**
 * اعلان درخواست تلگرام (alias)
 */
function tgRequest(string $method, array $params = []): ?array
{
    return sendRequest($method, $params);
}

// ─────────────────────────────────────────────────────────
// 👤 توابع کمکی کاربر
// ─────────────────────────────────────────────────────────

/**
 * ساخت منشن HTML برای کاربر
 */
function getUserMention(array $user): string
{
    $name = htmlspecialchars($user['first_name'] ?? 'کاربر');
    $userId = $user['id'];
    return "<a href='tg://user?id={$userId}'>{$name}</a>";
}

/**
 * دریافت نام کاربر
 */
function getUserName(array $user): string
{
    return $user['first_name'] ?? 'کاربر';
}

/**
 * دریافت نام کامل کاربر
 */
function getUserFullName(array $user): string
{
    $first = $user['first_name'] ?? '';
    $last = $user['last_name'] ?? '';
    return trim("{$first} {$last}") ?: 'کاربر';
}

// ─────────────────────────────────────────────────────────
// 🏢 توابع مدیریت گروه
// ─────────────────────────────────────────────────────────

/**
 * بررسی وضعیت گروه هنگام اضافه شدن ربات
 */
function checkGroupStatus(int $groupId, string $title): void
{
    $db = Database::instance()->pdo();
    
    // بررسی وجود گروه در دیتابیس
    $stmt = $db->prepare("SELECT * FROM groups_data WHERE group_id = ?");
    $stmt->execute([$groupId]);
    $group = $stmt->fetch();
    
    if (!$group) {
        // گروه جدید - ثبت اولیه
        $stmt = $db->prepare("
            INSERT INTO groups_data (group_id, group_title, is_active, created_at)
            VALUES (?, ?, 0, NOW())
        ");
        $stmt->execute([$groupId, $title]);
        
        // ارسال پیام خوش‌آمدگویی
        $welcomeText = "🎉 <b>به ربات خان بابا خوش آمدید!</b>\n\n";
        $welcomeText .= "⚠️ ربات هنوز فعال نیست.\n";
        $welcomeText .= "برای فعال‌سازی از دکمه زیر یا کد گیفت استفاده کنید.\n\n";
        $welcomeText .= "📌 دستور فعال‌سازی با کد گیفت:\n";
        $welcomeText .= "<code>!گیفت [کد]</code>";
        
        sendMessage($groupId, $welcomeText, [
            'inline_keyboard' => [
                [['text' => '📨 درخواست فعال‌سازی', 'callback_data' => 'request_install']]
            ]
        ]);
    } else {
        // گروه موجود - بررسی اعتبار
        if ($group['is_active']) {
            $expireDate = strtotime($group['expire_date']);
            $daysLeft = floor(($expireDate - time()) / 86400);
            
            $text = "✅ <b>ربات خان بابا دوباره فعال است!</b>\n\n";
            $text .= "⏳ اعتبار باقیمانده: {$daysLeft} روز\n";
            $text .= "📅 انقضا: " . jdate('Y/m/d H:i', $expireDate);
            
            sendMessage($groupId, $text);
        } else {
            $text = "⚠️ <b>ربات غیرفعال است</b>\n\n";
            $text .= "برای فعال‌سازی مجدد، از دکمه زیر استفاده کنید.";
            
            sendMessage($groupId, $text, [
                'inline_keyboard' => [
                    [['text' => '📨 درخواست فعال‌سازی', 'callback_data' => 'request_install']]
                ]
            ]);
        }
    }
}

/**
 * فعال‌سازی گروه
 */
function activateGroup(int $groupId, int $days, string $title): void
{
    $db = Database::instance()->pdo();
    $expireDate = date('Y-m-d H:i:s', strtotime("+{$days} days"));
    
    $stmt = $db->prepare("
        INSERT INTO groups_data (group_id, group_title, is_active, expire_date, updated_at)
        VALUES (?, ?, 1, ?, NOW())
        ON DUPLICATE KEY UPDATE
            is_active = 1,
            expire_date = VALUES(expire_date),
            group_title = VALUES(group_title),
            updated_at = NOW()
    ");
    $stmt->execute([$groupId, $title, $expireDate]);
    
    // ارسال پیام فعال‌سازی به گروه
    $text = "✅ <b>ربات با موفقیت فعال شد!</b>\n\n";
    $text .= "⏳ مدت اعتبار: {$days} روز\n";
    $text .= "📅 تاریخ انقضا: " . jdate('Y/m/d', strtotime($expireDate)) . "\n\n";
    $text .= "🎉 از تمامی امکانات ربات استفاده کنید!";
    
    sendMessage($groupId, $text);
    
    // اطلاع به سودو
    $sudoText = "✅ <b>گروه جدید فعال شد</b>\n\n";
    $sudoText .= "👥 نام: " . htmlspecialchars($title) . "\n";
    $sudoText .= "🆔 ID: <code>{$groupId}</code>\n";
    $sudoText .= "⏳ مدت: {$days} روز\n";
    $sudoText .= "📅 انقضا: " . jdate('Y/m/d', strtotime($expireDate));
    
    sendMessage(SUDO_ID, $sudoText);
}

/**
 * غیرفعال کردن گروه
 */
function deactivateGroup(int $groupId): void
{
    $db = Database::instance()->pdo();
    
    $stmt = $db->prepare("
        UPDATE groups_data 
        SET is_active = 0, updated_at = NOW()
        WHERE group_id = ?
    ");
    $stmt->execute([$groupId]);
}

/**
 * حذف کامل اطلاعات گروه
 */
function deleteGroupData(int $groupId): void
{
    $db = Database::instance()->pdo();
    
    // حذف از جداول مختلف
    $tables = [
        'groups_data',
        'group_admins',
        'mir_owners',
        'mir_deputies',
        'mir_advisors',
        'mir_admins',
        'mir_vips',
        'mir_banned',
        'mir_muted',
        'mir_warns',
        'mir_permissions',
        'mir_temp_data',
        'mir_users',
        'mia_locks',
        'me_users',
        'me_open_panels'
    ];
    
    foreach ($tables as $table) {
        try {
            $stmt = $db->prepare("DELETE FROM {$table} WHERE group_id = ?");
            $stmt->execute([$groupId]);
        } catch (PDOException $e) {
            // جدول وجود ندارد - ادامه
            continue;
        }
    }
}

// ─────────────────────────────────────────────────────────
// 🎁 توابع کد گیفت
// ─────────────────────────────────────────────────────────

/**
 * تولید کد گیفت جدید
 */
function generateGiftCode(int $days, int $creatorId): string
{
    $db = Database::instance()->pdo();
    
    // تولید کد یونیک
    do {
        $code = 'GIFT_' . strtoupper(substr(md5(uniqid() . rand()), 0, 12));
        
        $stmt = $db->prepare("SELECT 1 FROM gift_codes WHERE code = ?");
        $stmt->execute([$code]);
        $exists = $stmt->fetch();
    } while ($exists);
    
    // ذخیره در دیتابیس
    $stmt = $db->prepare("
        INSERT INTO gift_codes (code, days, is_used, created_by, created_at)
        VALUES (?, ?, 0, ?, NOW())
    ");
    $stmt->execute([$code, $days, $creatorId]);
    
    return $code;
}

/**
 * بررسی اعتبار کد گیفت
 */
function validateGiftCode(string $code): ?array
{
    $db = Database::instance()->pdo();
    
    $stmt = $db->prepare("
        SELECT * FROM gift_codes 
        WHERE code = ? AND is_used = 0
    ");
    $stmt->execute([$code]);
    
    return $stmt->fetch() ?: null;
}

/**
 * استفاده از کد گیفت
 */
function useGiftCode(string $code, int $groupId): bool
{
    $db = Database::instance()->pdo();
    
    $stmt = $db->prepare("
        UPDATE gift_codes 
        SET is_used = 1, used_by_group = ?, used_at = NOW()
        WHERE code = ? AND is_used = 0
    ");
    $stmt->execute([$groupId, $code]);
    
    return $stmt->rowCount() > 0;
}

// ─────────────────────────────────────────────────────────
// 📝 توابع لاگ
// ─────────────────────────────────────────────────────────

/**
 * ثبت لاگ در فایل
 */
function logMessage(string $message, string $type = 'info'): void
{
    $logDir = __DIR__ . '/logs';
    
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logFile = $logDir . '/bot_' . date('Y-m-d') . '.log';
    $timestamp = date('H:i:s');
    $logLine = "[{$timestamp}] [{$type}] {$message}\n";
    
    file_put_contents($logFile, $logLine, FILE_APPEND);
}

/**
 * ثبت خطا
 */
function logError(string $message, ?Throwable $exception = null): void
{
    $errorMsg = $message;
    
    if ($exception !== null) {
        $errorMsg .= " | File: " . $exception->getFile();
        $errorMsg .= " | Line: " . $exception->getLine();
        $errorMsg .= " | Message: " . $exception->getMessage();
    }
    
    logMessage($errorMsg, 'error');
}

// ─────────────────────────────────────────────────────────
// 🛡️ توابع امنیتی
// ─────────────────────────────────────────────────────────

/**
 * اعتبارسنجی ورودی
 */
function sanitizeInput(string $input): string
{
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * بررسی دسترسی سودو
 */
function isSudo(int $userId): bool
{
    return $userId === SUDO_ID;
}

/**
 * بررسی ادمین بودن در گروه
 */
function isGroupAdmin(int $groupId, int $userId): bool
{
    $db = Database::instance()->pdo();
    
    $stmt = $db->prepare("
        SELECT 1 FROM group_admins 
        WHERE group_id = ? AND user_id = ?
    ");
    $stmt->execute([$groupId, $userId]);
    
    return $stmt->fetch() !== false;
}

/**
 * بررسی سازنده گروه
 */
function isGroupCreator(int $groupId, int $userId): bool
{
    $db = Database::instance()->pdo();
    
    $stmt = $db->prepare("
        SELECT 1 FROM group_admins 
        WHERE group_id = ? AND user_id = ? AND is_creator = 1
    ");
    $stmt->execute([$groupId, $userId]);
    
    return $stmt->fetch() !== false;
}

// ─────────────────────────────────────────────────────────
// 🔧 توابع کمکی متفرقه
// ─────────────────────────────────────────────────────────

/**
 * فرمت کردن عدد با جداکننده
 */
function formatNumber(int $number): string
{
    return number_format($number);
}

/**
 * محاسبه زمان نسبی (مثل "2 ساعت پیش")
 */
function timeAgo(string $datetime): string
{
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'لحظاتی پیش';
    } elseif ($diff < 3600) {
        $minutes = floor($diff / 60);
        return "{$minutes} دقیقه پیش";
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return "{$hours} ساعت پیش";
    } else {
        $days = floor($diff / 86400);
        return "{$days} روز پیش";
    }
}

/**
 * تبدیل بایت به واحد قابل خواندن
 */
function formatBytes(int $bytes): string
{
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;
    
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    
    return round($bytes, 2) . ' ' . $units[$i];
}

?>