<?php
/**
 * فایل Cron Job - بروزرسانی شده
 * این فایل را هر ساعت یکبار با Cron Job اجرا کنید
 * مثال: 0 * * * * /usr/bin/php /path/to/cron.php >> /path/to/logs/cron.log 2>&1
 */

require_once 'config.php';
require_once 'jdate.php';
require_once 'me.php';

// تنظیم timezone
date_default_timezone_set('Asia/Tehran');

// لاگ شروع
logMessage("========================================");
logMessage("🔄 شروع بررسی گروه‌ها");
logMessage("⏱ زمان: " . date('Y-m-d H:i:s'));

/**
 * بررسی گروه‌های در حال انقضا
 */
function checkExpiringGroups() {
    $db = Database::instance()->pdo();
    
    // گروه‌هایی که 1 روز یا کمتر به انقضا مانده
    $stmt = $db->prepare("
        SELECT * FROM groups_data 
        WHERE is_active = 1 
        AND expire_date IS NOT NULL 
        AND expire_date <= DATE_ADD(NOW(), INTERVAL 1 DAY)
        AND expire_date > NOW()
    ");
    $stmt->execute();
    $groups = $stmt->fetchAll();
    
    logMessage("⏰ تعداد گروه‌های در حال انقضا: " . count($groups));
    
    foreach ($groups as $group) {
        $expire_timestamp = strtotime($group['expire_date']);
        $remaining = $expire_timestamp - time();
        $days_remaining = floor($remaining / 86400);
        $hours_remaining = floor($remaining / 3600);
        
        $text = "⚠️ <b>هشدار انقضای اشتراک!</b>\n\n";
        $text .= "اعتبار ربات در این گروه رو به پایان است.\n\n";
        
        if ($days_remaining > 0) {
            $text .= "⏳ زمان باقی‌مانده: {$days_remaining} روز\n";
        } else {
            $text .= "⏳ زمان باقی‌مانده: {$hours_remaining} ساعت\n";
        }
        
        $expire_jdate = jdate('Y/m/d H:i', $expire_timestamp);
        $text .= "📅 تاریخ انقضا: {$expire_jdate}\n\n";
        $text .= "برای تمدید اشتراک:\n";
        $text .= "• دستور: <code>!شارژ</code>\n";
        $text .= "• یا با مدیریت تماس بگیرید\n\n";
        $text .= "⚠️ پس از انقضا، تمامی امکانات غیرفعال می‌شوند!";
        
        $msg = sendMessage($group['group_id'], $text);
        
        // پین کردن پیام
        if ($msg && isset($msg['result']['message_id'])) {
            sendRequest('pinChatMessage', [
                'chat_id' => $group['group_id'],
                'message_id' => $msg['result']['message_id'],
                'disable_notification' => false
            ]);
            
            logMessage("✅ هشدار برای گروه " . $group['group_id'] . " ارسال شد");
        } else {
            logMessage("❌ خطا در ارسال هشدار برای گروه " . $group['group_id']);
        }
    }
    
    return count($groups);
}

/**
 * بررسی گروه‌های منقضی شده
 */
function checkExpiredGroups() {
    $db = Database::instance()->pdo();
    
    $stmt = $db->prepare("
        SELECT * FROM groups_data 
        WHERE is_active = 1 
        AND expire_date IS NOT NULL 
        AND expire_date <= NOW()
    ");
    $stmt->execute();
    $groups = $stmt->fetchAll();
    
    logMessage("🗑 تعداد گروه‌های منقضی شده: " . count($groups));
    
    foreach ($groups as $group) {
        // غیرفعال کردن گروه
        $stmt = $db->prepare("UPDATE groups_data SET is_active = 0 WHERE group_id = ?");
        $stmt->execute([$group['group_id']]);
        
        $text = "⏰ <b>اعتبار ربات به پایان رسید!</b>\n\n";
        $text .= "❌ تمامی امکانات غیرفعال شدند.\n\n";
        $text .= "برای تمدید اشتراک با مدیریت تماس بگیرید.\n";
        $text .= "دستور تمدید: <code>!شارژ</code>\n\n";
        $text .= "👋 ربات از گروه خارج می‌شود...";
        
        sendMessage($group['group_id'], $text);
        
        // صبر 3 ثانیه و ترک گروه
        sleep(3);
        $leave_result = sendRequest('leaveChat', ['chat_id' => $group['group_id']]);
        
        // اطلاع به سودو در PV
        $expire_jdate = jdate('Y/m/d H:i', strtotime($group['expire_date']));
        
        $notify = "📊 <b>گروه منقضی شده</b>\n\n";
        $notify .= "👥 <b>نام گروه:</b> " . htmlspecialchars($group['group_title']) . "\n";
        $notify .= "🆔 <b>شناسه:</b> <code>{$group['group_id']}</code>\n";
        $notify .= "📅 <b>تاریخ انقضا:</b> {$expire_jdate}\n\n";
        $notify .= "✅ ربات از گروه خارج شد.";
        
        sendMessage(SUDO_ID, $notify);
        
        if ($leave_result && $leave_result['ok']) {
            logMessage("✅ ربات از گروه " . $group['group_id'] . " خارج شد");
        } else {
            logMessage("⚠️ خطا در خروج از گروه " . $group['group_id']);
        }
    }
    
    return count($groups);
}

/**
 * پاکسازی لاگ‌های قدیمی
 */
function cleanOldLogs() {
    if (!file_exists('logs')) return;
    
    $log_files = glob('logs/*.log');
    $deleted = 0;
    
    foreach ($log_files as $file) {
        // حذف لاگ‌های بیش از 30 روز قدیمی
        if (filemtime($file) < time() - (30 * 24 * 60 * 60)) {
            if (unlink($file)) {
                $deleted++;
            }
        }
    }
    
    if ($deleted > 0) {
        logMessage("🗑 تعداد {$deleted} فایل لاگ قدیمی حذف شد");
    }
}

/**
 * گزارش آمار
 */
function generateStats() {
    $db = Database::instance()->pdo();
    
    // تعداد گروه‌های فعال
    $stmt = $db->query("SELECT COUNT(*) as count FROM groups_data WHERE is_active = 1");
    $active_groups = $stmt->fetch()['count'];
    
    // تعداد کل گروه‌ها
    $stmt = $db->query("SELECT COUNT(*) as count FROM groups_data");
    $total_groups = $stmt->fetch()['count'];
    
    // تعداد کدهای گیفت استفاده نشده
    $stmt = $db->query("SELECT COUNT(*) as count FROM gift_codes WHERE is_used = 0");
    $unused_gifts = $stmt->fetch()['count'];
    
    // تعداد کدهای گیفت استفاده شده
    $stmt = $db->query("SELECT COUNT(*) as count FROM gift_codes WHERE is_used = 1");
    $used_gifts = $stmt->fetch()['count'];
    
    logMessage("📊 آمار سیستم:");
    logMessage("  - گروه‌های فعال: {$active_groups}");
    logMessage("  - کل گروه‌ها: {$total_groups}");
    logMessage("  - کدهای گیفت استفاده نشده: {$unused_gifts}");
    logMessage("  - کدهای گیفت استفاده شده: {$used_gifts}");
    
    return [
        'active_groups' => $active_groups,
        'total_groups' => $total_groups,
        'unused_gifts' => $unused_gifts,
        'used_gifts' => $used_gifts
    ];
}

/**
 * ارسال گزارش روزانه به PV سودو
 */
function sendDailyReport() {
    // فقط یک بار در روز ساعت 9 صبح
    $current_hour = (int)date('H');
    if ($current_hour !== 9) return;
    
    $stats = generateStats();
    $db = Database::instance()->pdo();
    
    // لیست گروه‌هایی که امروز منقضی می‌شوند
    $stmt = $db->prepare("
        SELECT * FROM groups_data 
        WHERE is_active = 1 
        AND DATE(expire_date) = CURDATE()
    ");
    $stmt->execute();
    $expiring_today = $stmt->fetchAll();
    
    $report = "📊 <b>گزارش روزانه سیستم</b>\n";
    $report .= "📅 تاریخ: " . jdate('Y/m/d') . "\n\n";
    $report .= "📈 <b>آمار کلی:</b>\n";
    $report .= "• گروه‌های فعال: {$stats['active_groups']}\n";
    $report .= "• کل گروه‌ها: {$stats['total_groups']}\n";
    $report .= "• کدهای گیفت باقیمانده: {$stats['unused_gifts']}\n";
    $report .= "• کدهای گیفت استفاده شده: {$stats['used_gifts']}\n\n";
    
    if (count($expiring_today) > 0) {
        $report .= "⚠️ <b>گروه‌های منقضی امروز:</b>\n";
        foreach ($expiring_today as $group) {
            $report .= "• " . htmlspecialchars($group['group_title']) . " (<code>{$group['group_id']}</code>)\n";
        }
    } else {
        $report .= "✅ هیچ گروهی امروز منقضی نمی‌شود";
    }
    
    sendMessage(SUDO_ID, $report);
    logMessage("📧 گزارش روزانه به سودو ارسال شد");
}

/**
 * تابع لاگ
 */
function logMessage($message) {
    $log_file = 'logs/cron_' . date('Y-m-d') . '.log';
    
    // ایجاد پوشه logs در صورت عدم وجود
    if (!file_exists('logs')) {
        mkdir('logs', 0755, true);
    }
    
    $timestamp = date('H:i:s');
    file_put_contents($log_file, "[{$timestamp}] {$message}\n", FILE_APPEND);
    
    // نمایش در خروجی (برای cron log)
    echo "[{$timestamp}] {$message}\n";
}

// ═══════════════════════════════════════════════════════
// اجرای تسک‌ها
// ═══════════════════════════════════════════════════════

try {
    // 1. بررسی گروه‌های در حال انقضا
    logMessage("⏰ بررسی گروه‌های در حال انقضا...");
    $expiring_count = checkExpiringGroups();
    logMessage("✅ {$expiring_count} هشدار ارسال شد");
    
    // 2. بررسی گروه‌های منقضی شده
    logMessage("");
    logMessage("🗑 بررسی گروه‌های منقضی شده...");
    $expired_count = checkExpiredGroups();
    logMessage("✅ {$expired_count} گروه پردازش شد");
    
    // 3. بستن پنل‌های منقضی شده (سیستم ME)
    logMessage("");
    logMessage("⏰ بستن پنل‌های منقضی شده...");
    $closed_panels = closeExpiredPanels();
    logMessage("✅ {$closed_panels} پنل بسته شد");
    
    // 4. تولید آمار
    logMessage("");
    generateStats();
    
    // 5. ارسال گزارش روزانه به PV
    logMessage("");
    sendDailyReport();
    
    // 6. پاکسازی لاگ‌های قدیمی
    logMessage("");
    logMessage("🧹 پاکسازی لاگ‌های قدیمی...");
    cleanOldLogs();
    
    logMessage("");
    logMessage("🎉 تمام تسک‌ها با موفقیت انجام شد!");
    logMessage("⏱ زمان پایان: " . date('Y-m-d H:i:s'));
    logMessage("========================================");
    
    exit(0);
    
} catch (Exception $e) {
    logMessage("");
    logMessage("❌ خطای کلی: " . $e->getMessage());
    logMessage("📍 فایل: " . $e->getFile());
    logMessage("📍 خط: " . $e->getLine());
    logMessage("========================================");
    
    // اطلاع به سودو در PV در صورت خطا
    $error_msg = "🚨 <b>خطا در Cron Job</b>\n\n";
    $error_msg .= "⏱ زمان: " . date('Y-m-d H:i:s') . "\n";
    $error_msg .= "❌ پیام خطا: " . $e->getMessage() . "\n";
    $error_msg .= "📍 فایل: " . basename($e->getFile()) . "\n";
    $error_msg .= "📍 خط: " . $e->getLine();
    
    sendMessage(SUDO_ID, $error_msg);
    
    exit(1);
}

?>